<?php

use Slim\App;
use Slim\Routing\RouteCollectorProxy;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;
use Slim\Factory\AppFactory;

use \App\Middleware\JWTAuthMiddleware;
use \App\Action\Token\TokenAction;
use \App\Action\Users\UsersAction;
use \App\Action\Locations\LocationsAction;
use \App\Action\Activos\ActivosAction;
use \App\Action\ActiveFields\ActiveFieldsAction;
use \App\Action\Version\VersionAction;
use \App\Action\Fields\FieldsAction;
use \App\Action\City\CityAction;
use \App\Action\Logger\LoggerAction;
use \App\Action\Headquarters\HeadquartersAction;
use \App\Action\Dependency\DependencyAction;
use \App\Action\Shopping\ShoppingAction;
use \App\Action\TypePurchase\TypePurchaseAction;
use \App\Action\Categories\CategoriesAction;
use \App\Action\Responsible\ResponsibleAction;
use \App\Action\ReportFormat\ReportFormatAction;
use \App\Action\AssetLog\AssetLogAction;

// url app
$app = AppFactory::create();
$app->setBasePath("/ApiDev/InafHub/api/v1");

return function (App $app) {

    $container = $app->getContainer();
    $secretKey = $container->get('secret_key');

    $app->options('/{routes:.*}', function (Request $request, Response $response) {
        // CORS Pre-Flight OPTIONS Request Handler
        return $response;
    });

    $app->group('/Authenticate', function(RouteCollectorProxy $group) {
        $group->get('/validate-token', TokenAction::class . ':ValidateToken');
    });
    
    // user route
    $app->group('/Assets', function(RouteCollectorProxy $group){
        $group->post('/insert', ActivosAction::class . ':InsertRecord');
        $group->put('/adjust-asset-location', ActivosAction::class . ':UpdateAssetLocation');
        $group->put('/UpdateGlobalOfficial/{idOld}/{idNew}', ActivosAction::class . ':UpdateGlobalOfficial');
        $group->get('/search-id-only', ActivosAction::class . ':SearchIdOnly');
        $group->delete('/DeleteRecord/{id}', ActivosAction::class . ':DeleteRecord');
        $group->put('/transfer-responsible/{responsibleId}/{newResponsibleId}', ActivosAction::class . ':TransferResponsible');
        
        $group->get('/search-by-cities/{cities}/{offset}/{limit}', ActivosAction::class . ':SearchRecord');
        $group->get('/search-count-assets/{cities}', ActivosAction::class . ':CountAssets');
        $group->get('/search-by-id/{assetId}/{cities}', ActivosAction::class . ':SearchById');
        $group->get('/search-by-id-res/{responsibleId}', ActivosAction::class . ':SearchByIdRes');
        $group->put('/update-responsible/{assetId}/{responsibleId}', ActivosAction::class . ':UpdateFoKeyResponsible');
        $group->delete('/delete/{assetId}', ActivosAction::class . ':DeleteRecord');
        $group->put('/transfer-asset/{assetId}/{responsibleId}/{locationId}', ActivosAction::class . ':TransferAsset');
        $group->get('/asset-with-fields/{assetId}', ActivosAction::class . ':assetCategoryWithFields');
        $group->put('/update', ActivosAction::class . ':UpdateRecord');
        $group->put('/change-status/{assetId}/{status}', ActivosAction::class . ':ChangeAssetStatus');
        $group->get('/report-format/{cities}/{locationId}/{responsibleId}', ActivosAction::class . ':ReportFormat');
        $group->get('/search-by-responsible/{cities}/{responsibleId}', ActivosAction::class . ':SearchbyResponsible');
        $group->get('/search-by-description/{cities}/{data}', ActivosAction::class . ':SearchByDescription');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/Fields', function(RouteCollectorProxy $group){
        $group->get('/Search', FieldsAction::class . ':SearchRecords');
        $group->get('/search-by-id-category/{categoryId}', FieldsAction::class . ':FindCatId');
         

        $group->post('/insert', FieldsAction::class . ':InsertRecord');
        $group->put('/update', FieldsAction::class . ':UpdateRecord');
        $group->delete('/delete/{fieldId}', FieldsAction::class . ':DeleteRecord'); 
        $group->get('/search-with-status/{categoryId}', FieldsAction::class . ':SearchWithStatusByFoKey'); 
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/Users', function(RouteCollectorProxy $group){
        $group->post('/insert', UsersAction::class . ':InsertRecord');
        $group->get('/search', UsersAction::class . ':SearchRecords');
        $group->get('/SearchById/{email}/{pass}', UsersAction::class . ':SearchRecord');
        $group->put('/update', UsersAction::class . ':UpdateRecord');
        $group->delete('/delete/{id}', UsersAction::class . ':DeleteRecord');  
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/User', function(RouteCollectorProxy $group) use ($secretKey) {
        $group->get('/authenticate-user/{email}/{pass}', UsersAction::class . ':LoginVerify');

        // put cities to user email
        $group->put('/update-cities/{email}/{cities}/{city}', UsersAction::class . ':UpdateCities')
            ->add(new JWTAuthMiddleware($secretKey));
    });

    $app->group('/Locations', function(RouteCollectorProxy $group) {
        
        $group->get('/Search', LocationsAction::class . ':SearchRecords');
        $group->get('/search-by-id-city/{cityId}', LocationsAction::class . ':RecordsInCity');
        $group->get('/GetItems/{fk_id}', LocationsAction::class . ':GetItemsForLocat');
        $group->get('/SearchById/{data1}/{data2}', LocationsAction::class . ':SearchRecord');

        $group->post('/insert', LocationsAction::class . ':InsertRecord');
        $group->put('/update', LocationsAction::class . ':UpdateRecord');
        $group->delete('/delete/{locationId}', LocationsAction::class . ':DeleteRecord');
        $group->get('/search-with-status/{headquartersId}', LocationsAction::class . ':SearchWithStatusByFoKey');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/AssetField', function(RouteCollectorProxy $group) {
        $group->post('/insert', ActiveFieldsAction::class . ':InsertRecord');
        $group->get('/Search', ActiveFieldsAction::class . ':SearchRecords');
        $group->get('/SearchById/{data1}/{data2}', ActiveFieldsAction::class . ':SearchRecord');
        $group->put('/update', ActiveFieldsAction::class . ':UpdateRecord');
        $group->delete('/Delete/{id}', ActiveFieldsAction::class . ':DeleteRecord');

        $group->put('/update-desc', ActiveFieldsAction::class . ':UpdateDesc');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/Version', function(RouteCollectorProxy $group) {
        $group->get('/getAppV', VersionAction::class . ':getAppV');
    });

    $app->group('/City', function(RouteCollectorProxy $group) {

        $group->get('/search', CityAction::class . ':SearchRecords');
        $group->get('/search-by-ids/{cityId}', CityAction::class . ':SearchRecord');
        $group->post('/insert', CityAction::class . ':InsertRecord');
        $group->put('/update', CityAction::class . ':UpdateRecord');
        $group->delete('/delete/{cityId}', CityAction::class . ':DeleteRecord');

    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/Logger', function(RouteCollectorProxy $group) {
        $group->get('/search', LoggerAction::class . ':SearchRecords');
        $group->post('/insert', LoggerAction::class . ':InsertRecord');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/Headquarters', function(RouteCollectorProxy $group) {
        $group->post('/insert', HeadquartersAction::class . ':InsertRecord');
        $group->put('/update', HeadquartersAction::class . ':UpdateRecord');
        $group->get('/search', HeadquartersAction::class . ':SearchRecords');
        $group->get('/search-by-id-city/{cityId}', HeadquartersAction::class . ':SearchRecord');
        $group->get('/search-by-city-status/{cityId}', HeadquartersAction::class . ':SearchByCitiesWithStatus');
        $group->get('/search-join-city-status/{cityId}', HeadquartersAction::class . ':SearchJoinCityWithStatus');
        $group->put('/update-foreign-city', HeadquartersAction::class . ':UpdateFoKeyCity');
        $group->delete('/delete/{headquartersId}', HeadquartersAction::class . ':DeleteRecord');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/Dependency', function(RouteCollectorProxy $group) {
        $group->post('/insert', DependencyAction::class . ':InsertRecord');
        $group->put('/update', DependencyAction::class . ':UpdateRecord');
        $group->delete('/delete/{dependencyId}', DependencyAction::class . ':DeleteRecord');
        $group->get('/search', DependencyAction::class . ':SearchRecords');
        $group->get('/search-by-id-city/{cityId}', DependencyAction::class . ':SearchRecord');
        $group->get('/search-status-parent/{citiesId}', DependencyAction::class . ':SearchWithParentStatus');
        $group->get('/search-status-city/{citiesId}', DependencyAction::class . ':SearchWithStatusCity');
        $group->put('/update-foreign-city', DependencyAction::class . ':UpdateFoKeyCity');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/Shopping', function(RouteCollectorProxy $group) {
        $group->post('/insert', ShoppingAction::class . ':InsertRecord');
        $group->put('/update', ShoppingAction::class . ':UpdateRecord');
        $group->delete('/delete/{shoppingId}', ShoppingAction::class . ':DeleteRecord');
        $group->get('/city-data-no-pdf/{citiesId}', ShoppingAction::class . ':RecordsWithoutPDF');
        $group->get('/search', ShoppingAction::class . ':SearchRecords');
        $group->post('/upload-pdf-file/{shoppingId}', ShoppingAction::class . ':UploadFile');
        $group->get('/download-file-pdf/{shoppingId}', ShoppingAction::class . ':DownloadPDF');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/TypePurchase', function(RouteCollectorProxy $group) {
        $group->post('/insert', TypePurchaseAction::class . ':InsertRecord');
        $group->put('/update', TypePurchaseAction::class . ':UpdateRecord');
        $group->get('/search', TypePurchaseAction::class . ':SearchRecords');
        $group->delete('/delete/{purchaseId}', TypePurchaseAction::class . ':DeleteRecord');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/Category', function(RouteCollectorProxy $group) {
        $group->post('/insert', CategoriesAction::class . ':InsertRecord');
        $group->put('/update', CategoriesAction::class . ':UpdateRecord');
        $group->delete('/delete/{categoryId}', CategoriesAction::class . ':DeleteRecord'); 
        $group->get('/search', CategoriesAction::class . ':SearchRecords');
        $group->get('/search-by-id/{categoryId}', CategoriesAction::class . ':SearchRecord');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/Responsible', function(RouteCollectorProxy $group) {
        $group->get('/search', ResponsibleAction::class . ':SearchRecords');
        $group->delete('/delete/{responsibleId}', ResponsibleAction::class . ':DeleteRecord');
        $group->put('/update', ResponsibleAction::class . ':UpdateRecord');
        $group->post('/insert', ResponsibleAction::class . ':InsertRecord');
        $group->get('/select-with-assets/{responsibleId}', ResponsibleAction::class . ':SelectWithCountAsset');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/ReportFormat', function(RouteCollectorProxy $group) {
        $group->get('/asset-report-format/{formatId}', ReportFormatAction::class . ':SearchFormat');
    })->add(new JWTAuthMiddleware($secretKey));

    $app->group('/AssetLog', function(RouteCollectorProxy $group) {
        $group->get('/search', AssetLogAction::class . ':SearchRecords');
    })->add(new JWTAuthMiddleware($secretKey));
};